#include "json.hpp"
#include <stdexcept>
#include <cctype>

JSON::JSON():value_type(null_type) {}
JSON::JSON(nullptr_t):value_type(null_type) {}
JSON::JSON(bool v):value_type(bool_type),
   bool_value(v) {}
JSON::JSON(double v):value_type(number_type),
   number_value(v) {}
JSON::JSON(int v):value_type(number_type),
   number_value(v) {}
JSON::JSON(const string &v):value_type(string_type),
   string_value(v) {}
JSON::JSON(const char v[]):value_type(string_type),
   string_value(v) {}
JSON::JSON(const array &v):value_type(array_type),
   array_value(v) {}
JSON::JSON(const object &v):value_type(object_type),
   object_value(v) {}
JSON::JSON(const JSON &v):value_type(v.value_type) {
   switch (value_type) {
   case bool_type:
      bool_value=v.bool_value;
      break;
   case number_type:
      number_value=v.number_value;
      break;
   case string_type:
      string_value=v.string_value;
      break;
   case array_type:
      array_value=v.array_value;
      break;
   case object_type:
      object_value=v.object_value;
      break;
   }
}
void JSON::clear() {
   switch (value_type) {
   case string_type:
      string_value.clear();
      break;
   case array_type:
      array_value.clear();
      break;
   case object_type:
      object_value.clear();
      break;
   }
   value_type=null_type;
}
JSON::JSON_type JSON::type() const {
   return value_type;
}
JSON &JSON::operator=(const JSON &v) {
   clear();
   value_type=v.value_type;
   switch (value_type) {
   case bool_type:
      bool_value=v.bool_value;
      break;
   case number_type:
      number_value=v.number_value;
      break;
   case string_type:
      string_value=v.string_value;
      break;
   case array_type:
      array_value=v.array_value;
      break;
   case object_type:
      object_value=v.object_value;
      break;
   }
   return *this;
}
void JSON::push(const JSON &v) {
   if (value_type!=array_type) throw invalid_argument("Not array");
   array_value.push_back(v);
}
void JSON::pop() {
   if (value_type!=array_type) throw invalid_argument("Not array");
   if (array_value.size()==0) throw invalid_argument("Empty array");
   array_value.pop_back();
}
JSON &JSON::operator[](int i) {
   if (value_type!=array_type) throw invalid_argument("Not array");
   if (i<0) throw out_of_range("Negative index");
   if (i>=array_value.size())
      array_value.resize(i+1,JSON());
   return array_value[i];
}
const JSON &JSON::operator[](int i) const {
   if (value_type!=array_type) throw invalid_argument("Not array");
   if (i<0 || i>=array_value.size()) throw out_of_range("Wrong index");
   return array_value[i];
}
int JSON::length() const {
   if (value_type!=array_type) throw invalid_argument("Not array");
   return array_value.size();
}

JSON &JSON::operator[](const string &s) {
   if (value_type!=object_type) throw invalid_argument("Not object");
   return object_value[s];
}
const JSON &JSON::operator[](const string &s) const {
   if (value_type!=object_type) throw invalid_argument("Not object");
   return object_value.at(s);
}
JSON &JSON::operator[](const char s[]) {
   return operator[](string(s));
}
const JSON &JSON::operator[](const char s[]) const {
   return operator[](string(s));
}
bool JSON::has(const string &s) const {
   if (value_type!=object_type) throw invalid_argument("Not object");
   return object_value.find(s)!=object_value.end();
}
void JSON::erase(const string &s) {
   if (value_type!=object_type) throw invalid_argument("Not object");
   object_value.erase(s);
}
JSON::operator bool() const {
   switch (value_type) {
   case null_type:
      return false;
   case bool_type:
      return bool_value;
   case number_type:
      return number_value!=0;
   case string_type:
      return string_value.size()!=0;
   default:
      return true;
   }
}
JSON::operator double() const {
   if (value_type!=number_type) throw invalid_argument("Not number");
   return number_value;
}
JSON::operator const char *() const {
   if (value_type!=string_type) throw invalid_argument("Not string");
   return string_value.c_str();
}
JSON::operator string() const {
   if (value_type!=string_type) throw invalid_argument("Not string");
   return string_value;
}
static void out_string(ostream &o, const string &s) {
   o<<'"';
   for (auto &c:s) {
      if (c=='"' || c=='\\') o<<'\\'<<c;
      else if (c=='\n') o<<"\\n";
      else if (c=='\r') o<<"\\r";
      else if (c=='\t') o<<"\\t";
      else o<<c;
   }
   o<<'"';
}
ostream &operator<<(ostream &o, const JSON &v) {
   switch (v.value_type) {
   case JSON::null_type:
      o<<"null";
      break;
   case JSON::bool_type:
      o<<(v.bool_value?"true":"false");
      break;
   case JSON::number_type:
      o<<v.number_value;
      break;
   case JSON::string_type:
      out_string(o,v.string_value);
      break;
   case JSON::array_type:
      o<<"[";
      for (int i=0; i<v.array_value.size(); ++i) {
         if (i>0) o<<", ";
         o<<v.array_value[i];
      }
      o<<"]";
      break;
   case JSON::object_type:
      o<<"{";
      bool flg=false;
      for (auto &x: v.object_value) {
         if (flg) o<<", ";
         out_string(o, x.first);
         o<<": ";
         o<<x.second;
         flg=true;
      }
      o<<"}";
      break;
   }
   return o;
}
static bool in_string(istream &i, string &s) {
   s.clear();
   char c;
   while (1) {
      if (!i.get(c)) return false;
      if (c=='"') break;
      if (c=='\\') {
         if(!i.get(c)) return false;
         if (c=='\\' || c=='"') s+=c;
         else if (c=='n') s+='\n';
         else if (c=='r') s+='\r';
         else if (c=='t') s+='\t';
      }
      else s+=c;
   }
   return true;
}
istream &operator>>(istream &i, JSON &v) {
   v.clear();
   char c;
   string key;
   JSON j;
   i.setf(ios::skipws);
   if (i>>c) {
      if (isalpha(c))
      { key=c;
         while(1) {
            int k=i.peek();
            if(k<0 || !isalpha(char(k))) break;
            i.get(c);
            key+=c;
         }
         if (key=="null") ;
         else if (key=="true" || key=="false") {
            v.value_type=JSON::bool_type;
            v.bool_value= key=="true";
         }
         else i.clear(ios::failbit);
      }
      else if (isdigit(c)) {
         i.unget();
         v.value_type=JSON::number_type;
         i>>v.number_value;
      }
      else if (c=='"') {
         v.value_type=JSON::string_type;
         in_string(i,v.string_value);
      }
      else if (c=='[') {
         v.value_type=JSON::array_type;
         if (i>>c)
            if (c!=']') {
               i.unget();
               while (1) {
                  if (i>>j && i>>c) {
                     v.array_value.push_back(j);
                     if (c==',') ;
                     else if (c==']') break;
                     else {
                        i.clear(ios::failbit);
                        break;
                     }
                  }
                  else
                     break;
               }
            }
      }
      else if (c=='{') {
         v.value_type=JSON::object_type;
         if (i>>c)
            if (c!=']') {
               i.unget();
               while (1) {
                  if (i>>c && c=='"' && in_string(i,key) &&
                        i>>c && c==':' && i>>j && i>>c && (c=='}' || c==',')) {
                     v.object_value[key]=j;
                     if (c=='}') break;
                  }
                  else {
                     i.clear(ios::failbit);
                     break;
                  }
               }
            }
      }
      else {
         i.clear(ios::failbit);
      }
   }
   return i;
}
